/*
 * Copyright (c) 2003, 2022, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package sun.management;

import java.lang.reflect.Method;
import java.lang.reflect.Field;
import java.lang.management.ManagementPermission;
import java.lang.management.ThreadInfo;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.List;
import javax.management.ObjectName;
import javax.management.MalformedObjectNameException;


public class Util {
    private Util() {}  // there are no instances of this class

    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    static String[] toStringArray(List<String> list) {
        return list.toArray(EMPTY_STRING_ARRAY);
    }

    public static ObjectName newObjectName(String domainAndType, String name) {
        return newObjectName(domainAndType + ",name=" + name);
    }

    public static ObjectName newObjectName(String name) {
        try {
            return ObjectName.getInstance(name);
        } catch (MalformedObjectNameException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static ManagementPermission monitorPermission =
        new ManagementPermission("monitor");
    private static ManagementPermission controlPermission =
        new ManagementPermission("control");

    /**
     * Check that the current context is trusted to perform monitoring
     * or management.
     * <p>
     * If the check fails we throw a SecurityException, otherwise
     * we return normally.
     *
     * @exception  SecurityException  if a security manager exists and if
     *             the caller does not have ManagementPermission("control").
     */
    static void checkAccess(ManagementPermission p)
         throws SecurityException {
        @SuppressWarnings("removal")
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            sm.checkPermission(p);
        }
    }

    static void checkMonitorAccess() throws SecurityException {
        checkAccess(monitorPermission);
    }
    public static void checkControlAccess() throws SecurityException {
        checkAccess(controlPermission);
    }

    /**
     * Returns true if the given Thread is a virtual thread.
     *
     * @implNote This method uses reflection because Thread::isVirtual is a preview API
     * and the java.management module cannot be compiled with --enable-preview.
     */
    public static boolean isVirtual(Thread thread) {
        try {
            return (boolean) THREAD_IS_VIRTUAL.invoke(thread);
        } catch (Exception e) {
            throw new InternalError(e);
        }
    }

    /**
     * Returns true if the given ThreadInfo is for a virtual thread.
     */
    public static boolean isVirtual(ThreadInfo threadInfo) {
        try {
            return (boolean) THREADINFO_VIRTUAL.get(threadInfo);
        } catch (Exception e) {
            throw new InternalError(e);
        }
    }

    @SuppressWarnings("removal")
    private static Method threadIsVirtual() {
        PrivilegedExceptionAction<Method> pa = () -> Thread.class.getMethod("isVirtual");
        try {
            return AccessController.doPrivileged(pa);
        } catch (PrivilegedActionException e) {
            throw new InternalError(e);
        }
    }

    @SuppressWarnings("removal")
    private static Field threadInfoVirtual() {
        PrivilegedExceptionAction<Field> pa = () -> {
            Field f = ThreadInfo.class.getDeclaredField("virtual");
            f.setAccessible(true);
            return f;
        };
        try {
            return AccessController.doPrivileged(pa);
        } catch (PrivilegedActionException e) {
            throw new InternalError(e);
        }
    }

    private static final Method THREAD_IS_VIRTUAL = threadIsVirtual();
    private static final Field THREADINFO_VIRTUAL = threadInfoVirtual();
}
